<?php
/**
 * ColorMag Admin Class.
 *
 * @author  ThemeGrill
 * @package ColorMag
 * @since   1.1.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'ColorMag_Admin' ) ) :

	/**
	 * ColorMag_Admin Class.
	 */
	class ColorMag_Admin {

		/**
		 * Constructor.
		 */
		public function __construct() {
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
			add_action( 'load-toplevel_page_colormag', array( $this, 'handle_account_edits' ) );
		}

		/**
		 * Registers REST API routes for the ColorMag theme.
		 *
		 * This function creates an instance of the `Colormag_Changelog_Controller` class
		 * and calls its `register_routes` method to register the necessary REST API routes.
		 *
		 * @return void
		 */
		public function register_rest_routes() {
			register_rest_route(
				'colormag/v1',
				'/settings',
				array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'update_settings' ),
					'permission_callback' => function () {
						return current_user_can( 'manage_options' );
					},
				)
			);

			( new Colormag_Changelog_Controller() )->register_routes();
		}

		/**
		 * @param WP_REST_Request $request
		 */
		public function update_settings( $request ) {
			$params = $request->get_params();

			if ( isset( $params['enable_white_label'] ) ) {
				$params['enable_white_label'] = filter_var( $params['enable_white_label'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_starter_templates'] ) ) {
				$params['hide_starter_templates'] = filter_var( $params['hide_starter_templates'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_products_tab'] ) ) {
				$params['hide_products_tab'] = filter_var( $params['hide_products_tab'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_helps_tab'] ) ) {
				$params['hide_helps_tab'] = filter_var( $params['hide_helps_tab'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_license_tab'] ) ) {
				$params['hide_license_tab'] = filter_var( $params['hide_license_tab'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_documentation'] ) ) {
				$params['hide_documentation'] = filter_var( $params['hide_documentation'], FILTER_VALIDATE_BOOLEAN );
			}if ( isset( $params['hide_starter_template_section'] ) ) {
				$params['hide_starter_template_section'] = filter_var( $params['hide_starter_template_section'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_review_section'] ) ) {
				$params['hide_review_section'] = filter_var( $params['hide_review_section'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['hide_feature_section'] ) ) {
				$params['hide_feature_section'] = filter_var( $params['hide_feature_section'], FILTER_VALIDATE_BOOLEAN );
			}if ( isset( $params['hide_support_section'] ) ) {
				$params['hide_support_section'] = filter_var( $params['hide_support_section'], FILTER_VALIDATE_BOOLEAN );
			}if ( isset( $params['hide_user_ful_section'] ) ) {
				$params['hide_user_ful_section'] = filter_var( $params['hide_user_ful_section'], FILTER_VALIDATE_BOOLEAN );
			}if ( isset( $params['hide_community_section'] ) ) {
				$params['hide_community_section'] = filter_var( $params['hide_community_section'], FILTER_VALIDATE_BOOLEAN );
			}
			if ( isset( $params['agency_name'] ) ) {
				$params['agency_name'] = filter_var( $params['agency_name'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}if ( isset( $params['agency_url'] ) ) {
				$params['agency_url'] = filter_var( $params['agency_url'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}if ( isset( $params['theme_name'] ) ) {
				$params['theme_name'] = filter_var( $params['theme_name'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}if ( isset( $params['theme_description'] ) ) {
				$params['theme_description'] = filter_var( $params['theme_description'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}
			if ( isset( $params['theme_screenshot'] ) ) {
				$params['theme_screenshot'] = filter_var( $params['theme_screenshot'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}

			if ( isset( $params['theme_icon'] ) ) {
				$params['theme_icon'] = filter_var( $params['theme_icon'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}if ( isset( $params['dashboard_icon'] ) ) {
				$params['dashboard_icon'] = filter_var( $params['dashboard_icon'], FILTER_SANITIZE_FULL_SPECIAL_CHARS );
			}

			$settings = get_option( '_colormag_settings', array() );

			$settings = wp_parse_args( $params, $settings );

			update_option( '_colormag_settings', $settings );

			return rest_ensure_response( $request->get_params() );
		}

		/**
		 * Prepares data for Freemius integration.
		 *
		 * This function initializes the Freemius SDK and gathers various data points
		 * related to the plugin, user, site, and license. It also prepares hidden input
		 * fields for license synchronization, deactivation, and user information updates.
		 * The data is then returned in an array format, which can be used for AJAX requests
		 * and other operations within the admin interface.
		 *
		 * @return array|null The prepared data array or null if Freemius initialization fails.
		 */
		public function prepare_freemius_data() {
			$fs = FS_ThemeGrill::init(
				'4212',
				'colormag',
				'pk_414d89e1f7eda2dd7de41050ab418',
				'ColorMag'
			);

			if ( ! $fs ) {
				return null;
			}

			$data = array();
			if ( $fs->can_use_premium_code() && $fs->get_user() instanceof \FS_User ) {
				ob_start();
				?>
				<input type="hidden" name="fs_action" value="<?php echo esc_attr( $fs->get_unique_affix() ); ?>_sync_license">
				<?php
				wp_nonce_field( $fs->get_unique_affix() . '_sync_license' );
				$sync = ob_get_clean();

				ob_start();
				?>
				<input type="hidden" name="fs_action" value="deactivate_license">
				<?php
				wp_nonce_field( 'deactivate_license' );
				$deactivate_license = ob_get_clean();
				ob_start();
				?>
				<input type="hidden" name="fs_action" value="update_user_name">
				<input type="hidden" name="fs_user_name_colormag-theme" value="">
				<?php
				wp_nonce_field( 'update_user_name' );
				$user_name = ob_get_clean();
				?>

				<input type="hidden" name="fs_action" value="update_email">
				<input type="hidden" name="fs_email_colormag-theme" value="">
				<?php
				wp_nonce_field( 'update_email' );
				$email_address = ob_get_clean();

				/**
				 * @var \FS_Plugin_Tag $update
				 */
				$update = $fs->has_release_on_freemius() ?
					$fs->get_update( false, false ) :
					null;

				if ( is_object( $update ) ) {
					$updater = \FS_Plugin_Updater::instance( $fs );
					$updater->set_update_data( $update );
				}

				$user    = $fs->get_user();
				$site    = $fs->get_site();
				$name    = $user->get_name();
				$license = $fs->_get_license();
				$plan    = $fs->get_plan();

				$available_licenses = $fs->get_available_premium_licenses();
				$available_licenses = array_reduce(
					$available_licenses,
					function ( $carry, $license ) {
						$activations_left = $license->left();
						if ( ( $activations_left > 0 ) ) {
							$carry[ $activations_left . '_' . $license->id ] = $license;
						}
						return $carry;
					},
					array()
				);
				$available_license  = current( $available_licenses );

				if ( $available_license instanceof \FS_Plugin_License ) {
					$available_license_text = sprintf(
						'%s-Site %s License - %s',
						( 1 === $available_license->quota ?
							'Single' :
							( $available_license->is_unlimited() ? 'Unlimited' : $available_license->quota )
						),
						$fs->_get_plan_by_id( $available_license->plan_id )->title,
						$available_license->get_html_escaped_masked_secret_key()
					);
				} else {
					$available_license_text = 'No available license';
				}

				$has_bundle_license = false;

				if (
					is_object( $license ) &&
					\FS_Plugin_License::is_valid_id( $license->parent_license_id )
				) {
					$has_bundle_license = true;
					$bundle_plan_title  = strtoupper( $license->parent_plan_name );
				}

				$update = $fs->has_release_on_freemius() ?
					$fs->get_update( false, false ) :
					null;

				$billing = $fs->_fetch_billing();

				$has_billing = ( $billing instanceof \FS_Billing );
				if ( ! $has_billing ) {
					$billing = new \FS_Billing();
				}

				$context_params = array(
					'plugin_id'         => $fs->get_id(),
					'plugin_public_key' => $fs->get_public_key(),
					'plugin_version'    => $fs->get_plugin_version(),
				);

				if ( $fs->is_registered() ) {
					$context_params = array_merge(
						$context_params,
						\FS_Security::instance()->get_context_params(
							$fs->get_site(),
							time(),
							'contact'
						)
					);
				}

				$query_params = array_merge(
					$_GET, // phpcs:ignore
					array_merge(
						$context_params,
						array(
							'plugin_version' => $fs->get_plugin_version(),
							'wp_login_url'   => wp_login_url(),
							'site_url'       => \Freemius::get_unfiltered_site_url(),
						)
					)
				);

				$data = array(
					'activated' => true,
					'account'   => array(
						'data'                 => array(
							'user_name'              => $name,
							'email'                  => $user->email,
							'user_id'                => $user->id,
							'product'                => $fs->get_plugin_title(),
							'product_id'             => $fs->get_id(),
							'site_id'                => is_string( $site->id ) ?
								$site->id :
								__( 'No ID', 'colormag' ),
							'site_public_key'        => $site->public_key,
							'site_secret_key'        => ( ( is_string( $site->secret_key ) ) ?
								$site->secret_key :
								__( 'No Secret', 'colormag' )
							),
							'version'                => $fs->get_plugin_version(),
							'license_key'            => $license->secret_key,
							'available_license_text' => $available_license_text,
							'beta_program'           => ! fs_is_network_admin() && $fs->is_premium() && $site->is_beta,
							'plan'                   => strtoupper(
								is_string( $plan->name ) ?
									$plan->title :
									__( 'Free', 'colormag' ),
							),
							'bundle_plan'            => $has_bundle_license ? $bundle_plan_title : null,
							'update'                 => is_object( $update ) ? array(
								'update_url'     => wp_nonce_url( self_admin_url( "update.php?action=upgrade-{$fs->get_module_type()}&{$fs->get_module_type()}=" . $fs->get_plugin_basename() ), "upgrade-{$fs->get_module_type()}_" . $fs->get_plugin_basename() ),
								'update_version' => $update->version,
							) : null,
						),
						'sync'                 => $sync,
						'deactivate_license'   => $deactivate_license,
						'email'                => $email_address,
						'user_name'            => $user_name,
						'update_license'       => array(
							'action'   => $fs->get_ajax_action( 'activate_license' ),
							'security' => $fs->get_ajax_security( 'activate_license' ),
						),
						'update_email_address' => array(
							'action'   => $fs->get_ajax_action( 'update_email_address' ),
							'security' => $fs->get_ajax_security( 'update_email_address' ),
						),
						'set_beta_mode'        => array(
							'action'   => $fs->get_ajax_action( 'set_beta_mode' ),
							'security' => $fs->get_ajax_security( 'set_beta_mode' ),
						),
					),
					'billing'   => array(
						'payments' => array_reduce(
							$fs->_fetch_payments(),
							function ( $acc, $curr ) use ( $fs ) {
								$acc[] = array(
									'id'      => $curr->id,
									'date'    => $curr->created,
									'amount'  => $curr->formatted_gross(),
									'invoice' => ! $curr->is_migrated() ? $fs->_get_invoice_api_url( $curr->id ) : null,
								);
								return $acc;
							},
							array()
						),
						'action'   => $fs->get_ajax_action( 'update_billing' ),
						'security' => $fs->get_ajax_security( 'update_billing' ),
						'details'  => array_filter(
							(array) $billing,
							function ( $v ) {
								return ! is_null( $v ) && '' !== $v && 'null' !== $v;
							}
						),
					),
					'contact'   => array(
						'src'      => WP_FS__ADDRESS . '/contact/?' . http_build_query( $query_params ),
						'base_url' => WP_FS__ADDRESS,
					),
				);
			} else {
				$data = array(
					'activated'  => false,
					'activation' => array(
						'action'   => $fs->get_ajax_action( 'activate_license' ),
						'security' => $fs->get_ajax_security( 'activate_license' ),
					),
				);
			}

			$data['ajax_url']           = $fs::ajax_url( '' );
			$data['module_id']          = $fs->get_id();
			$data['resend_license_key'] = array(
				'action'   => $fs->get_ajax_action( 'resend_license_key' ),
				'security' => $fs->get_ajax_security( 'resend_license_key' ),
			);

			return $data;
		}

		/**
		 * Enqueues admin scripts and styles for the ColorMag theme.
		 *
		 * This function enqueues the necessary CSS and JavaScript files for the admin interface.
		 * It also localizes scripts with data required for AJAX requests and other operations.
		 * The function checks the current screen to ensure scripts are only loaded on relevant pages.
		 * Additionally, it prepares data for Freemius integration and localizes it for use in the dashboard.
		 *
		 * @return void
		 * Handles account edits for the ColorMag theme.
		 *
		 * This function initializes the Freemius SDK and uses reflection to access and invoke
		 * the private `_handle_account_edits` method of the Freemius object. It also enqueues
		 * necessary scripts for handling post messages and JSON operations.
		 *
		 * @return void
		 */
		public function handle_account_edits() {
			$fs = FS_ThemeGrill::init(
				'4212',
				'colormag',
				'pk_414d89e1f7eda2dd7de41050ab418',
				'ColorMag'
			);

			$_handle_account_edits = new \ReflectionMethod( $fs, '_handle_account_edits' );
			$_handle_account_edits->setAccessible( true );
			try {
				$_handle_account_edits->invoke( $fs );
			} catch ( ReflectionException $e ) {
				error_log( print_r( $e, true ) );
			}

			wp_enqueue_script( 'jquery' );
			wp_enqueue_script( 'json2' );
			fs_enqueue_local_script( 'postmessage', 'nojquery.ba-postmessage.js' );
			fs_enqueue_local_script( 'fs-postmessage', 'postmessage.js' );
		}

		/**
		 * Enqueues admin scripts and styles for the ColorMag theme.
		 *
		 * This function enqueues the necessary CSS and JavaScript files for the admin interface.
		 * It also localizes scripts with data required for AJAX requests and other operations.
		 * The function checks the current screen to ensure scripts are only loaded on relevant pages.
		 * Additionally, it prepares data for Freemius integration and localizes it for use in the dashboard.
		 *
		 * @return void
		 */
		public function enqueue_scripts() {
			wp_enqueue_style( 'colormag-admin-style', get_template_directory_uri() . '/inc/admin/css/admin.css', array(), COLORMAG_THEME_VERSION );
			wp_enqueue_script( 'colormag-plugin-install-helper', get_template_directory_uri() . '/inc/admin/js/plugin-handle.js', array( 'jquery' ), COLORMAG_THEME_VERSION, true );

			// Only expose the nonce and AJAX functionality to users with appropriate capabilities
			$welcome_data = array(
				'uri'       => esc_url( admin_url( '/themes.php?page=colormag&tab=starter-templates' ) ),
				'btn_text'  => esc_html__( 'Processing...', 'colormag' ),
				'admin_url' => esc_url( admin_url() ),
			);

			// Only add nonce and ajaxurl if user has appropriate capabilities
			if ( current_user_can( 'manage_options' ) ) {
				$welcome_data['nonce']   = wp_create_nonce( 'colormag_demo_import_nonce' );
				$welcome_data['ajaxurl'] = admin_url( 'admin-ajax.php' );
			}

			wp_localize_script( 'colormag-plugin-install-helper', 'colormagRedirectDemoPage', $welcome_data );

			// This ensures that the following code only runs on the relevant admin pages for the ColorMag theme dashboard.
			$screen = get_current_screen();
			if ( ! in_array( $screen->id, array( 'toplevel_page_colormag', 'colormag_page_colormag-starter-templates', 'appearance_page_colormag', 'appearance_page_colormag-starter-templates' ), true ) ) {
				return;
			}

			$build_dir_uri        = apply_filters( 'colormag_build_dir_uri', get_template_directory_uri() . '/assets/build/' );
			$build_dir_path       = apply_filters( 'colormag_build_dir_path', get_template_directory() . '/assets/build/' );
			$dashboard_asset_file = $build_dir_path . 'dashboard.asset.php';
			if ( file_exists( $dashboard_asset_file ) ) {
				$dashboard_asset = require $dashboard_asset_file;
				wp_enqueue_script( 'colormag-dashboard', $build_dir_uri . 'dashboard.js', $dashboard_asset['dependencies'], $dashboard_asset['version'], true );
				wp_enqueue_style( 'colormag-dashboard', $build_dir_uri . 'dashboard.css', array( 'wp-components' ), time() );
			}
			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			if ( ! function_exists( 'wp_get_themes' ) ) {
				require_once ABSPATH . 'wp-admin/includes/theme.php';
			}
			$installed_plugin_slugs = array_keys( get_plugins() );
			$allowed_plugin_slugs   = array(
				'everest-forms/everest-forms.php',
				'user-registration/user-registration.php',
				'learning-management-system/lms.php',
				'magazine-blocks/magazine-blocks.php',
				'themegrill-demo-importer/themegrill-demo-importer.php',
			);

			$installed_theme_slugs = array_keys( wp_get_themes() );
			$current_theme         = get_stylesheet();
			$freemius_data         = $this->prepare_freemius_data();

			//Dashboard script localization.
			$colormag_dashboard_setting = get_option( '_colormag_settings', array() );

			wp_localize_script( 'colormag-dashboard', '__COLORMAG_SETTINGS__', $colormag_dashboard_setting );

			// Localize the script for dashboard.
			$localized_scripts = apply_filters(
				'colormag_localize_admin_scripts',
				array(
					'name' => '_COLORMAG_DASHBOARD_',
					'data' => array(
						'version'       => COLORMAG_THEME_VERSION,
						'plugins'       => array_reduce(
							$allowed_plugin_slugs,
							function ( $acc, $curr ) use ( $installed_plugin_slugs ) {
								if ( in_array( $curr, $installed_plugin_slugs, true ) ) {
									if ( is_plugin_active( $curr ) ) {
										$acc[ $curr ] = 'active';
									} else {
										$acc[ $curr ] = 'inactive';
									}
								} else {
									$acc[ $curr ] = 'not-installed';
								}
								return $acc;
							},
							array()
						),
						'builder'       => colormag_maybe_enable_builder(),
						'demoUrl'       => admin_url( 'admin.php?page=colormag-starter-templates' ),
						'demoImporter'  => is_plugin_active( 'themegrill-demo-importer/themegrill-demo-importer.php' ) ? 'active' : 'inactive',
						'customizerUrl' => admin_url( 'customize.php' ),
						'dashboardLogo' => white_label_settings()->get( 'dashboard_icon' ) ? white_label_settings()->get( 'dashboard_icon' ) : '',
						'enableWhiteLabel'    => white_label_settings()->get( 'enable_white_label' ),
						'themeName'    => white_label_settings()->get( 'Name' ) ? white_label_settings()->get( 'Name' ) : 'ColorMag',
						'themes'        => array(
							'colormag' => strpos( $current_theme, 'colormag' ) !== false ? 'active' : (
							in_array( 'colormag', $installed_theme_slugs, true ) ? 'inactive' : 'not-installed'
							),
							'zakra'    => strpos( $current_theme, 'zakra' ) !== false || strpos( $current_theme, 'zakra' ) !== false ? 'active' : (
							in_array( 'zakra', $installed_theme_slugs, true ) || in_array( 'zakra', $installed_theme_slugs, true ) ? 'inactive' : 'not-installed'
							),
						),
						'adminUrl'      => admin_url(),
						'fs'            => $freemius_data,
						'userRoles'     => array_reduce(
							array_keys( wp_roles()->roles ),
							function ( $acc, $cur ) {
								$acc[ $cur ] = wp_roles()->roles[ $cur ]['name'];
								return $acc;
							},
							[]
						),
						'nonce'         => wp_create_nonce( 'colormag_dashboard_nonce' ),
						'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
						'popupEditUrl'  => admin_url( 'post.php?post=%d&action=edit' ),
					),
				)
			);

			$handle = apply_filters( 'colormag_localize_admin_scripts_handle', 'colormag-dashboard' );
			wp_localize_script( $handle, $localized_scripts['name'], $localized_scripts['data'] );
		}
	}

endif;

return new ColorMag_Admin();
